/* Stretch javascript code, taken from Apple examples */


/*
 * Stretcher constructor; parameters:
 *
 * -- element: The element to stretch
 * -- stretchDistance: Distance (in pixels) the content should stretch
 * -- stretchDuration: How long (in ms) the stretch animation should take
 * -- onFinished: A callback (if no callback is needed, pass null)
 *
 */
function Stretcher (element, stretchDistance, stretchDuration, onFinished) {
	this.element = element;

	this.startTime = 0;
	this.timer = null;
	
	this.duration = stretchDuration;
	this.multiplier = 1;
	this.stretchTime = 0;
	
	this.stretchDistance = stretchDistance;
	
	// min and max position can be changed to alter the stretched/shrunk sizes;
	// getComputedStyle depends on the target (in this case, the stretcher element)
	// being visible, so don't instantiate the Stretcher until the content is shown
	this.minPosition = parseInt(document.defaultView.getComputedStyle(this.element, "").getPropertyValue("height"));
	this.maxPosition = this.minPosition + this.stretchDistance;
	
	// Set variables to what they'd be in the beginning "shrunk" state
	this.positionFrom = this.minPosition;
	this.positionNow = this.minPosition;
	this.positionTo = this.minPosition;
		
	this.onFinished = onFinished;
}

/*
 * This should only be called via a Stretcher instance, i.e. "instance.stretch(event)"
 * Calling Stretcher_stretch() directly will result in "this" evaluating to the window
 * object, and the function will fail; parameters:
 * 
 * -- event: the mouse click that starts everything off (from an onclick handler)
 *		We check for the shift key to do a slo-mo stretch
 */
Stretcher.prototype.stretch = function (event) {
	if (event && event != undefined && event.shiftKey) {
		// enable slo-mo
		this.multiplier = 10;
	} else this.multiplier = 1;
	
	var timeNow = (new Date).getTime();
	
	if (this.timer != null) {
		// We're already stretching in some direction;
		// change the destination position and restart the timer
		clearInterval(this.timer);
		this.timer = null;
		this.stretchTime -= (timeNow - this.startTime);
		this.positionFrom = this.positionNow;
	} else {
		this.stretchTime = this.duration * this.multiplier;
		this.positionFrom = this.positionNow;
	}

	// Change from our previous direction
	if (this.positionTo == this.minPosition) {
		this.positionTo = this.maxPosition;
	} else {
		this.positionTo = this.minPosition;
	}

	// If we're expanding, resize the window to make room for the newly-sized content
	// We don't want to do this before shrinking because we'll clip the content
	// Check for the widget object because this will be annoying in Safari
	if (window.widget && (this.positionTo == this.maxPosition)) {
		window.resizeTo(parseInt(document.defaultView.getComputedStyle(this.element, "").getPropertyValue("width")), this.positionTo);
	}

	this.startTime = timeNow - 13; // set it back one frame.
		
	// We need to store this in a local variable so the timer
	// does not lose scope when invoking tick
	var localThis = this;
	this.tick();
	this.timer = setInterval (function() { localThis.tick(); }, 13);

}
		
/*
 * Tick does all the incremental resize work
 * This function is very similar to the tick() function in the Fader sample
 */
Stretcher.prototype.tick = function () {
	var T;
	var ease;
	var time  = (new Date).getTime();
	var frame;
		
	T = limit_3(time-this.startTime, 0, this.stretchTime);
	ease = 0.5 - (0.5 * Math.cos(Math.PI * T / this.stretchTime));

	if (T >= this.stretchTime) {
		// go to the finished position when the timer is up
		this.positionNow = this.positionTo;
		clearInterval (this.timer);
		this.timer = null;
		// If we're shrinking, we resize the window AFTER the animation is complete
		// Otherwise we'll clip the content as it shrinks
		if (window.widget && this.positionTo == this.minPosition) {
			window.resizeTo(parseInt(document.defaultView.getComputedStyle(this.element, "").getPropertyValue("width")), this.positionNow);
		}
		if (this.onFinished) {
			// call after the last frame is drawn
			var localThis = this;
			setTimeout (function() { localThis.onFinished(); }, 0);
		}
	} else {
		this.positionNow = parseInt(computeNextFloat(this.positionFrom, this.positionTo, ease));
	}

	this.element.style.height = this.positionNow + "px";
}

/*
 * Report whether or not the Stretcher is in its maximized position
 * DO NOT call this function to determine whether or not the Stretcher is 
 * currently animating; set the onFinished handler to be notified when animation
 * is complete
 */
Stretcher.prototype.isStretched = function() {
	return (this.positionNow == this.maxPosition);
}

/*
 * Support functions for the stretch animation
 */
function limit_3 (a, b, c) {
    return a < b ? b : (a > c ? c : a);
}

function computeNextFloat (from, to, ease) {
    return from + (to - from) * ease;
}